/*
 * CRfePUR.cpp
 *
 *  Created on: 11.05.2009
 *      Author: stefan.detter
 */

#include "../inc/reader/QrfeRfePurReader.h"
#include "../inc/epc/EPC_Defines.h"
#include <qmath.h>
#include <QVariant>
#include <QDataStream>

#include <QrfeProtocolHandlerRFE_Defines>

const QStringList QrfeRfePurReader::m_rssiChildNames = QStringList() << "Channel Q" << "Channel I";
const int QrfeRfePurReader::m_rssiChildMin = 0;
const int QrfeRfePurReader::m_rssiChildMax = 32;

QrfeRfePurReader::QrfeRfePurReader(QrfeProtocolHandlerRFE* ph, QObject* parent)
	: QrfeRfeReader(ph, parent)
{
    m_statusRegisterBitDescritption[STATUSBIT_AT91_SPURIOUS_BIT]    = "AT91_SPURIOUS_BIT";
    m_statusRegisterBitDescritption[STATUSBIT_EEPROM_STATUS_BIT]    = "EEPROM_STATUS_BIT";
    m_statusRegisterBitDescritption[STATUSBIT_ANTENNA_TO_HIGH_RFP]  = "ANTENNA_TO_HIGH_RFP";

	m_statusRegisterBitDescritption[STATUSBIT_AS399X_STATUS_BIT] 				= "AS399X_STATUS_BIT";
	m_statusRegisterBitDescritption[STATUSBIT_AS399X_COM_ERROR_BIT] 			= "AS399X_COM_ERROR_BIT";
	m_statusRegisterBitDescritption[STATUSBIT_AS399X_SET_FREQUENCY_PROBLEM] 	= "AS399X_SET_FREQUENCY_PROBLEM";
	m_statusRegisterBitDescritption[STATUSBIT_AS399X_WRONG_FREQUENCY_PROBLEM] 	= "AS399X_WRONG_FREQUENCY_PROBLEM";
	m_statusRegisterBitDescritption[STATUSBIT_AS399X_ANTENNA_SET_PROBLEM] 		= "AS399X_ANTENNA_SET_PROBLEM";
    m_statusRegisterBitDescritption[STATUSBIT_AS399X_ERROR_ON_INIT] 			= "AS399X_ERROR_ON_INIT";
    m_statusRegisterBitDescritption[STATUSBIT_AS399X_PLL_NOT_LOCKED] 			= "AS399X_PLL_NOT_LOCKED";


	m_statusRegisterBitDescritption[STATUSBIT_GEN2_CRC_ERROR_BIT] 				= "GEN2_CRC_ERROR";

	m_as399xResetCount = 0;

	m_retreivedValidG = false;
	m_as399xGconstant = 0;
}

QrfeRfePurReader::~QrfeRfePurReader()
{
	if(m_currentState != QrfeGlobal::STATE_OFFLINE){
		stopCyclicInventory();
		setAntennaPower(false);
	}
}


QrfeGlobal::Result QrfeRfePurReader::readSettings()
{
	QrfeGlobal::Result res;

    if( m_ph->getAntennaCount( m_antennaCount ) != QrfeGlobal::RES_OK)
        m_antennaCount = 1;

	bool rssiOn;
	res = rssiEnabled(rssiOn);
	if(res == QrfeGlobal::RES_OK)
	{
		emit changedRssiEnable(rssiOn, m_rssiChildNames.size(), m_rssiChildNames, m_rssiChildMin, m_rssiChildMax);
	}

	ushort mode;
	res = inventoryMode(mode);
	if(res == QrfeGlobal::RES_OK)
	{
		if( (m_readerTypeExact&0x000000FF) == 0x09)
			m_tagType = QrfeGlobal::TAG_ISO6B;
		else if(mode == 0x03)
			m_tagType = QrfeGlobal::TAG_GEN2_NXP_READPROTECTED;
		else
			m_tagType = QrfeGlobal::TAG_GEN2;
	}
	else
	{
		m_tagType = QrfeGlobal::TAG_GEN2;
	}


	if((m_softwareRevision&0x7FFF) >= 0x0109)
	{
		bool frequOn;
		res = readFrequencyEnabled(frequOn);
		if(res == QrfeGlobal::RES_OK)
		{
			emit changedReadFrequencyEnable(frequOn);
		}
	}
	else
		emit changedReadFrequencyEnable(false);

	res = as399xGconstant(m_as399xGconstant);
	if(res == QrfeGlobal::RES_OK)
	{
		m_retreivedValidG = true;
	}

	if ((m_softwareRevision & 0x7FFF) >= 0x0218)
	{
        m_ph->activateNotification(QrfeRfePurReader::NOTIFICATION_ID_ANTENNA_CHANNEL_CHANGED);
		m_ph->activateNotification(QrfeRfePurReader::NOTIFICATION_ID_ANTENNA_RFP_ERROR);
	}

    if ((m_softwareRevision & 0x7FFF) >= 0x0224)
    {
        m_ph->activateNotification(QrfeRfePurReader::NOTIFICATION_ID_DEBUG_CODES);
    }

	return res;
}


QStringList QrfeRfePurReader::rssiChildNames()
{
	return m_rssiChildNames;
}


QrfeGlobal::Result QrfeRfePurReader::inventoryMode(ushort &mode)
{
	QByteArray modeArray;
	QrfeGlobal::Result ret = getParam(0x00, modeArray);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(modeArray.size() != 1)
		return QrfeGlobal::RES_ERROR;

	mode = modeArray.at(0);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setInventoryMode(ushort mode)
{
	if(mode == 0x03)
		m_tagType = QrfeGlobal::TAG_GEN2_NXP_READPROTECTED;
	else
		m_tagType = QrfeGlobal::TAG_GEN2;

	emit tagTypeChanged(m_tagType);

	QByteArray modeArray;
	modeArray.append(mode);
	return setParam(0x00, modeArray);
}

QrfeGlobal::Result QrfeRfePurReader::powerSafe( bool &on, ushort &interval )
{
	QByteArray powerSafe;
	QrfeGlobal::Result ret = getParam(0x01, powerSafe);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(powerSafe.size() != 3)
		return QrfeGlobal::RES_ERROR;

	on = powerSafe.at(0) == 1;
	interval = ( (ushort)powerSafe.at(1) ) << 8;
	interval += (uchar)powerSafe.at(2);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setPowerSafe( bool on, ushort interval )
{
	QByteArray powerSafe;
	powerSafe.append(on);
	powerSafe.append(interval>>8);
	powerSafe.append(interval);
	return setParam(0x01, powerSafe);
}

QrfeGlobal::Result QrfeRfePurReader::powerSafeCount( ushort &count )
{
	QByteArray powerSafeCount;
	QrfeGlobal::Result ret = getParam(0x07, powerSafeCount);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(powerSafeCount.size() != 2)
		return QrfeGlobal::RES_ERROR;

	count = ( (ushort)powerSafeCount.at(0) ) << 8;
	count += (uchar)powerSafeCount.at(1);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setPowerSafeCount( ushort count )
{
	QByteArray powerSafe;
	powerSafe.append(count>>8);
	powerSafe.append(count);
	return setParam(0x07, powerSafe);
}

QrfeGlobal::Result QrfeRfePurReader::rssiEnabled ( bool &on )
{
	QByteArray rssi;
	QrfeGlobal::Result ret = getParam(0x02, rssi);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(rssi.size() != 1)
		return QrfeGlobal::RES_ERROR;

	on = rssi.at(0) == 1;

	emit changedRssiEnable(on, m_rssiChildNames.size(), m_rssiChildNames, m_rssiChildMin, m_rssiChildMax);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setRssiEnabled ( bool on )
{
	QByteArray rssi;
	rssi.append(on);
	QrfeGlobal::Result ret = setParam(0x02, rssi);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	emit changedRssiEnable(on, m_rssiChildNames.size(), m_rssiChildNames, m_rssiChildMin, m_rssiChildMax);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::tagIdHandler ( uchar &handler )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x03, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 1)
		return QrfeGlobal::RES_ERROR;

	handler = data.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setTagIdHandler ( uchar handler )
{
	QByteArray data;
	data.append(handler);
	QrfeGlobal::Result ret = setParam(0x03, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::readFrequencyEnabled ( bool &on )
{
	QByteArray payl;
	QrfeGlobal::Result ret = getParam(0x05, payl);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(payl.size() != 1)
		return QrfeGlobal::RES_ERROR;

	on = payl.at(0) == 1;

	emit changedReadFrequencyEnable(on);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setReadFrequencyEnabled ( bool on )
{
	QByteArray payl;
	payl.append(on);
	QrfeGlobal::Result ret = setParam(0x05, payl);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	emit changedReadFrequencyEnable(on);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::modulationDepth ( uchar &percent )
{
	QByteArray perc;
	QrfeGlobal::Result ret = getParam(0x22, perc);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(perc.size() != 1)
		return QrfeGlobal::RES_ERROR;

	percent = perc.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setModulationDepth ( uchar percent )
{
	QByteArray perc;
	perc.append(percent);
	QrfeGlobal::Result ret = setParam(0x22, perc);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}


QrfeGlobal::Result QrfeRfePurReader::linkFrequency ( uchar &lf )
{
	QByteArray link;
	QrfeGlobal::Result ret = getParam(0x20, link);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(link.size() != 1)
		return QrfeGlobal::RES_ERROR;

	lf = link.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setLinkFrequency ( uchar lf )
{
	QByteArray data;
	data.append(lf);
	QrfeGlobal::Result ret = setParam(0x20, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::coding ( uchar &coding )
{
	QByteArray code;
	QrfeGlobal::Result ret = getParam(0x21, code);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(code.size() != 1)
		return QrfeGlobal::RES_ERROR;

	coding = code.at(0);
	return ret;

}

QrfeGlobal::Result QrfeRfePurReader::setCoding ( uchar coding )
{
	QByteArray data;
	data.append(coding);
	QrfeGlobal::Result ret = setParam(0x21, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}


QrfeGlobal::Result QrfeRfePurReader::longPreamble ( bool &longPreamble )
{
    QByteArray code;
    QrfeGlobal::Result ret = getParam(0x2D, code);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    if(code.size() != 1)
        return QrfeGlobal::RES_ERROR;

    longPreamble = (code.at(0) == 0x01);
    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setLongPreamble ( bool longPreamble )
{
    QByteArray data;
    data.append( (longPreamble) ? 0x01 : 0x00);
    QrfeGlobal::Result ret = setParam(0x2D, data);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    return ret;
}



QrfeGlobal::Result QrfeRfePurReader::epcSize ( uchar &epcSize )
{
	QByteArray size;
	QrfeGlobal::Result ret = getParam(0x23, size);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(size.size() != 1)
		return QrfeGlobal::RES_ERROR;

	epcSize = size.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setEpcSize ( uchar epcSize )
{
	QByteArray data;
	data.append(epcSize);
	QrfeGlobal::Result ret = setParam(0x23, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::sendRN16( bool &on )
{
    QByteArray once;
    QrfeGlobal::Result ret = getParam(0x2E, once);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    if(once.size() != 1)
        return QrfeGlobal::RES_ERROR;

    on = once.at(0) == 1;
    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setSendRN16( bool on )
{
    QByteArray once;
    once.append(on);
    QrfeGlobal::Result ret = setParam(0x2E, once);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::sendHandle( bool &on )
{
	QByteArray once;
	QrfeGlobal::Result ret = getParam(0x24, once);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(once.size() != 1)
		return QrfeGlobal::RES_ERROR;

	on = once.at(0) == 1;
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setSendHandle( bool on )
{
	QByteArray once;
	once.append(on);
	QrfeGlobal::Result ret = setParam(0x24, once);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::sendPC ( bool &on )
{
	QByteArray once;
	QrfeGlobal::Result ret = getParam(0x25, once);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(once.size() != 1)
		return QrfeGlobal::RES_ERROR;

	on = once.at(0) == 1;
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setSendPC ( bool on )
{
	QByteArray once;
	once.append(on);
	QrfeGlobal::Result ret = setParam(0x25, once);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::postDetectReadParams ( bool &on, uchar &memBank, ushort &address, uchar &size, QByteArray& password )
{
	QByteArray postDetectREAD;
	QrfeGlobal::Result ret = getParam(0x06, postDetectREAD);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(postDetectREAD.size() != 9)
		return QrfeGlobal::RES_ERROR;

	on 			= (postDetectREAD.at(0) != 0);
	memBank 	= postDetectREAD.at(1);
	address 	= 0;
	address		+= (ushort)( ((uchar)postDetectREAD.at(2)) << 8);
	address		+= (ushort)( ((uchar)postDetectREAD.at(3)) << 0);
	size	 	= postDetectREAD.at(4);
	password 	= postDetectREAD.mid(5, 4);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setPostDetectReadParams ( bool on, uchar memBank, ushort address, uchar size, QByteArray password )
{
	QByteArray postDetectREAD;
	postDetectREAD.append(on);
	postDetectREAD.append(memBank);
	postDetectREAD.append((uchar)(address >> 8));
	postDetectREAD.append((uchar)(address >> 0));
	postDetectREAD.append(size);
	postDetectREAD.append(password);

	QrfeGlobal::Result ret = setParam(0x06, postDetectREAD);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::q( uchar &initialQ, uchar &minimalQ, uchar &maximalQ )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x26, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 3)
		return QrfeGlobal::RES_ERROR;

	initialQ = data.at(0);
	minimalQ = data.at(1);
	maximalQ = data.at(2);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setQ ( uchar initialQ, uchar minimalQ, uchar maximalQ )
{
	QByteArray data;
	data.append(initialQ);
	data.append(minimalQ);
	data.append(maximalQ);
	QrfeGlobal::Result ret = setParam(0x26, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::qMethod( uchar &qMethod )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x27, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 1)
		return QrfeGlobal::RES_ERROR;

	qMethod = data.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setQMethod ( uchar qMethod )
{
	QByteArray data;
	data.append(qMethod);
	QrfeGlobal::Result ret = setParam(0x27, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::session( uchar &session )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x28, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 1)
		return QrfeGlobal::RES_ERROR;

	session = data.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setSession ( uchar session )
{
	QByteArray data;
	data.append(session);
	QrfeGlobal::Result ret = setParam(0x28, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::inventoryRounds( uchar &invRounds )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x29, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 1)
		return QrfeGlobal::RES_ERROR;

	invRounds = data.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setInventoryRounds ( uchar invRounds )
{
	QByteArray data;
	data.append(invRounds);
	QrfeGlobal::Result ret = setParam(0x29, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::selectionMask1( uchar &memBank, ulong& address, uchar& size, QByteArray& mask )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x2A, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() < 6)
		return QrfeGlobal::RES_ERROR;

	memBank 	= data.at(0);
	address 	= 0;
	address 	+= ( (ulong)(uchar)data.at(1) ) << 24;
	address 	+= ( (ulong)(uchar)data.at(2) ) << 16;
	address 	+= ( (ulong)(uchar)data.at(3) ) << 8;
	address 	+= (ulong)(uchar)data.at(4);
	size 		= data.at(5);

	ulong byteCount 	= QrfeGlobal::bitCountTobyteCount(size);
	if(data.size() != (6 + byteCount))
		return QrfeGlobal::RES_ERROR;

	mask 		= data.mid(6, byteCount);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setSelectionMask1( uchar memBank, ulong address, uchar size, const QByteArray mask )
{
	QByteArray data;
	data.append(memBank);
	data.append((uchar) (address >> 24));
	data.append((uchar) (address >> 16));
	data.append((uchar) (address >> 8));
	data.append((uchar) (address >> 0));
	data.append(size);
	data.append(mask);
	QrfeGlobal::Result ret = setParam(0x2A, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::selectionMask2( uchar &memBank, ulong& address, uchar& size, QByteArray& mask )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x2B, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() < 6)
		return QrfeGlobal::RES_ERROR;

	memBank 	= data.at(0);
	address 	= 0;
	address 	+= ( (ulong)(uchar)data.at(1) ) << 24;
	address 	+= ( (ulong)(uchar)data.at(2) ) << 16;
	address 	+= ( (ulong)(uchar)data.at(3) ) << 8;
	address 	+= (ulong)(uchar)data.at(4);
	size 		= data.at(5);

	ulong byteCount 	= QrfeGlobal::bitCountTobyteCount(size);
	if(data.size() != (6 + byteCount))
		return QrfeGlobal::RES_ERROR;

	mask 		= data.mid(6, byteCount);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setSelectionMask2( uchar memBank, ulong address, uchar size, const QByteArray mask )
{
	QByteArray data;
	data.append(memBank);
	data.append((uchar) (address >> 24));
	data.append((uchar) (address >> 16));
	data.append((uchar) (address >> 8));
	data.append((uchar) (address >> 0));
	data.append(size);
	data.append(mask);
	QrfeGlobal::Result ret = setParam(0x2B, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::queryNxpBrandID ( bool &on )
{
    QByteArray once;
    QrfeGlobal::Result ret = getParam(0x2F, once);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    if(once.size() != 1)
        return QrfeGlobal::RES_ERROR;

    on = once.at(0) == 1;
    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setQueryNxpBrandID ( bool on )
{
    QByteArray once;
    once.append(on);
    QrfeGlobal::Result ret = setParam(0x2F, once);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::as399xGconstant( uchar &G )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0x0012, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 1)
		return QrfeGlobal::RES_ERROR;

	G = data.at(0);
	return ret;
}


QrfeGlobal::Result QrfeRfePurReader::setTestFrequency(uint freq)
{
	QByteArray frequency;
	frequency.append((freq >> 16));
	frequency.append((freq >> 8));
	frequency.append((freq));
	return setParam(0x10, frequency);
}

QrfeGlobal::Result QrfeRfePurReader::measureReflectedPower( uint freq, ushort time, char &chI, char &chQ, uchar &G )
{
	QByteArray testResult;

	if( (m_softwareRevision & 0x7FFF) < 0x0107)
	{
		QrfeGlobal::Result ret;
		ret = setTestFrequency(freq);
		if(ret != QrfeGlobal::RES_OK)
			return ret;

		ret = getParam(0x10, testResult);
		if(ret != QrfeGlobal::RES_OK)
			return ret;
	}
	else
	{
		QByteArray frequency;
		frequency.append((freq >> 16));
		frequency.append((freq >> 8));
		frequency.append((freq));

		if( (m_softwareRevision & 0x7FFF) >= 0x0108)
		{
			frequency.append((time >> 8));
			frequency.append((time));
		}

		QrfeGlobal::Result ret = m_ph->executeAnalysisFunction(0x01, frequency, testResult);
		handleProtocolHandlerError(ret);
		if(ret != QrfeGlobal::RES_OK)
			return ret;
	}


	if(testResult.size() != 3)
		return QrfeGlobal::RES_ERROR;

	chI  = testResult.at(0);
	chQ  = testResult.at(1);
	G    = testResult.at(2);

	return QrfeGlobal::RES_OK;
}

QrfeGlobal::Result QrfeRfePurReader::measureRSSILevels ( uint freq, ushort time, uchar &rawQ, uchar &rawI, char &dBm )
{
	QByteArray testResult;
	if( (m_softwareRevision & 0x7FFF) < 0x0107)
	{
		QrfeGlobal::Result ret;
		ret = setTestFrequency(freq);
		if(ret != QrfeGlobal::RES_OK)
			return ret;

		ret = getParam(0x11, testResult);
		if(ret != QrfeGlobal::RES_OK)
			return ret;
	}
	else
	{
		QByteArray frequency;
		frequency.append((freq >> 16));
		frequency.append((freq >> 8));
		frequency.append((freq));

		if( (m_softwareRevision & 0x7FFF) >= 0x0108)
		{
			frequency.append((time >> 8));
			frequency.append((time));
		}

		QrfeGlobal::Result ret = m_ph->executeAnalysisFunction(0x02, frequency, testResult);
		handleProtocolHandlerError(ret);
		if(ret != QrfeGlobal::RES_OK)
			return ret;
	}

	if(testResult.size() != 3)
		return QrfeGlobal::RES_ERROR;

	rawQ = testResult.at(0);
	rawI = testResult.at(1);
	dBm  = testResult.at(2);

	return QrfeGlobal::RES_OK;
}



QrfeGlobal::Result QrfeRfePurReader::as399xRegister( QByteArray &regs )
{
	QrfeGlobal::Result ret;

	if( (m_softwareRevision & 0x7FFF) < 0x0108)
		ret = getParam(0x40, regs);
	else
		ret = getParam(0xFF00, regs);

	if(ret != QrfeGlobal::RES_OK)
		return ret;

	return QrfeGlobal::RES_OK;
}

QrfeGlobal::Result QrfeRfePurReader::minimumAttenuation ( uchar &minAtt )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0xFF01, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 1)
		return QrfeGlobal::RES_ERROR;

	minAtt = data.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setMinimumAttenuation ( uchar minAtt )
{
	QByteArray data;
	data.append(minAtt);
	return setParam(0xFF01, data);
}

QrfeGlobal::Result QrfeRfePurReader::frequencyOffset ( long &offset )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0xFF02, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 3)
		return QrfeGlobal::RES_ERROR;

	offset = 0;
	offset += (((long)((uchar)data.at(0))) << 16);
	offset += (((long)((uchar)data.at(1))) << 8);
	offset += (((long)((uchar)data.at(2))) << 0);

	if((data.at(0) & 0x80) > 0)
		offset += (((long)(char)0xFF) << 24);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setFrequencyOffset ( long offset )
{
	QByteArray data;
	data.append((char)( ((ulong)offset) >> 16));
	data.append((char)( ((ulong)offset) >> 8));
	data.append((char)( ((ulong)offset) >> 0));
	return setParam(0xFF02, data);
}

QrfeGlobal::Result QrfeRfePurReader::defaultFrequencyListIndex ( uchar &index )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0xFF03, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 1)
		return QrfeGlobal::RES_ERROR;

	index = data.at(0);
	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setDefaultFrequencyListIndex ( uchar index )
{
	QByteArray data;
	data.append(index);
	return setParam(0xFF03, data);
}

QrfeGlobal::Result QrfeRfePurReader::defaultModulatorSettings ( ulong &modSet )
{
	QByteArray data;
	QrfeGlobal::Result ret = getParam(0xFF04, data);
	if(ret != QrfeGlobal::RES_OK)
		return ret;

	if(data.size() != 4)
		return QrfeGlobal::RES_ERROR;

	modSet = 0;
	modSet += (((ulong)((uchar)data.at(0))) << 24);
	modSet += (((ulong)((uchar)data.at(1))) << 16);
	modSet += (((ulong)((uchar)data.at(2))) << 8);
	modSet += (((ulong)((uchar)data.at(3))) << 0);

	return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setDefaultModulatorSettings ( ulong modSet )
{
	QByteArray data;
	data.append((char)( ((ulong)modSet) >> 24));
	data.append((char)( ((ulong)modSet) >> 16));
	data.append((char)( ((ulong)modSet) >> 8));
	data.append((char)( ((ulong)modSet) >> 0));
	return setParam(0xFF04, data);
}

QrfeGlobal::Result QrfeRfePurReader::maxAntennaCount(uchar &count)
{
    QByteArray data;
    QrfeGlobal::Result ret = getParam(0xFF05, data);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    if(data.size() != 1)
        return QrfeGlobal::RES_ERROR;

    count = data.at(0);
    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setAntennaCount ( uchar count )
{
    QByteArray data;
    data.append(count);

    QrfeGlobal::Result ret = setParam(0xFF05, data);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    emit signatureChanged();

    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::autoReflectedPowerMeasurementData ( uchar &time, uchar &cycles, short &threshold )
{
    QByteArray data;
    QrfeGlobal::Result ret = getParam(0xFF06, data);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    if(data.size() != 4)
        return QrfeGlobal::RES_ERROR;

    time = data.at(0);
    cycles = data.at(1);
    threshold = 0;
    threshold += (((short)((uchar)data.at(2))) << 8);
    threshold += (((short)((uchar)data.at(3))) << 0);

    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setAutoReflectedPowerMeasurementData ( uchar time, uchar cycles, short threshold )
{
    QByteArray data;
    data.append(time);
    data.append(cycles);
    data.append((char)( ((ulong)threshold) >> 8));
    data.append((char)( ((ulong)threshold) >> 0));
    return setParam(0xFF06, data);

}

QrfeGlobal::Result QrfeRfePurReader::powerSafeRequiredSettings ( bool &required, ushort &minTime, ushort &maxCount)
{
    QByteArray data;
    QrfeGlobal::Result ret = getParam(0xFF07, data);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    if(data.size() != 5)
        return QrfeGlobal::RES_ERROR;

    required = (data.at(0) != 0);
    minTime = 0;
    minTime += (((ushort)((uchar)data.at(1))) << 8);
    minTime += (((ushort)((uchar)data.at(2))) << 0);
    maxCount = 0;
    maxCount += (((ushort)((uchar)data.at(3))) << 8);
    maxCount += (((ushort)((uchar)data.at(4))) << 0);

    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setPowerSafeRequiredSettings ( bool required, ushort minTime, ushort maxCount )
{
    QByteArray data;
    data.append((required)?1:0);
    data.append((char)( ((ulong)minTime) >> 8));
    data.append((char)( ((ulong)minTime) >> 0));
    data.append((char)( ((ulong)maxCount) >> 8));
    data.append((char)( ((ulong)maxCount) >> 0));
    return setParam(0xFF07, data);
}

QrfeGlobal::Result QrfeRfePurReader::referenceFrequency(uchar &refFreq)
{
    QByteArray data;
    QrfeGlobal::Result ret = getParam(0xFF08, data);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    if(data.size() != 1)
        return QrfeGlobal::RES_ERROR;

    refFreq = data.at(0);
    return ret;
}

QrfeGlobal::Result QrfeRfePurReader::setReferenceFrequency ( uchar refFreq )
{
    QByteArray data;
    data.append(refFreq);

    QrfeGlobal::Result ret = setParam(0xFF08, data);
    if(ret != QrfeGlobal::RES_OK)
        return ret;

    return ret;
}


QrfeGlobal::TMIResult QrfeRfePurReader::readFromTag ( const QString& tagId, uchar mem_bank, ushort address, const QByteArray& passwd, uchar count, QByteArray &data )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	//m_ph->setAntennaPower(true);
	QrfeGlobal::Result res = QrfeGlobal::RES_ERROR;
	for(int i = 0; i < 3; i++){
		res = m_ph->readFromTag(epc, mem_bank, address, passwd, count, data);
		QrfeGlobal::TMIResult tmi_res = translateGen2ProtocolHandlerResult(res);
		if(tmi_res == QrfeGlobal::TMI_OK || tmi_res == QrfeGlobal::TMI_MEM_LOCKED || tmi_res == QrfeGlobal::TMI_MEM_OVERRUN || tmi_res == QrfeGlobal::TMI_WRONG_PASSW )
			break;
	}
	//m_ph->setAntennaPower(false);
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::readMultipleFromTag ( const QString& tagId, QList<uchar> mem_bank, QList<ushort> address, QList<QByteArray> passwd, QList<uchar> count, QList<QByteArray> &data )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	//m_ph->setAntennaPower(true);
	QrfeGlobal::Result res = QrfeGlobal::RES_ERROR;
	for(int i = 0; i < 3; i++){
		res = m_ph->readMultipleFromTag(epc, mem_bank, address, passwd, count, data);
		QrfeGlobal::TMIResult tmi_res = translateGen2ProtocolHandlerResult(res);
		if(tmi_res == QrfeGlobal::TMI_OK || tmi_res == QrfeGlobal::TMI_MEM_LOCKED || tmi_res == QrfeGlobal::TMI_MEM_OVERRUN || tmi_res == QrfeGlobal::TMI_WRONG_PASSW )
			break;
	}
	//m_ph->setAntennaPower(false);
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}


QrfeGlobal::TMIResult QrfeRfePurReader::writeToTag (const QString& tagId, uchar mem_bank, ushort address, const QByteArray& passwd, const QByteArray& data , bool retry)
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	//m_ph->setAntennaPower(true);
	QrfeGlobal::Result res = QrfeGlobal::RES_ERROR;
	for(int i = 0; i < 3; i++){
		res = m_ph->writeToTag(epc, mem_bank, address, passwd, data);
		QrfeGlobal::TMIResult tmi_res = translateGen2ProtocolHandlerResult(res);
		if(tmi_res == QrfeGlobal::TMI_OK || tmi_res == QrfeGlobal::TMI_MEM_LOCKED || tmi_res == QrfeGlobal::TMI_MEM_OVERRUN || tmi_res == QrfeGlobal::TMI_WRONG_PASSW )
			break;
        if(!retry)
            break;
	}
	//m_ph->setAntennaPower(false);
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}


QrfeGlobal::TMIResult QrfeRfePurReader::lockTag ( const QString& tagId, uchar mode, uchar memory, const QByteArray& passwd )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	//m_ph->setAntennaPower(true);
	QrfeGlobal::Result res = QrfeGlobal::RES_ERROR;
	for(int i = 0; i < 3; i++){
		res = m_ph->lockTag(epc, mode, memory, passwd);
		QrfeGlobal::TMIResult tmi_res = translateGen2ProtocolHandlerResult(res);
		if(tmi_res == QrfeGlobal::TMI_OK || tmi_res == QrfeGlobal::TMI_MEM_LOCKED || tmi_res == QrfeGlobal::TMI_MEM_OVERRUN || tmi_res == QrfeGlobal::TMI_WRONG_PASSW )
			break;
	}
	//m_ph->setAntennaPower(false);
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::killTag ( const QString& tagId, const QByteArray& killPassword )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	QrfeGlobal::Result res = m_ph->killTag(epc, 0, 0, killPassword);
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}


QrfeGlobal::TMIResult QrfeRfePurReader::blockWriteToTag ( const QByteArray& tagId, uchar mem_bank, ushort address, const QByteArray& passwd, uchar blockSize, const QByteArray& data )
{
	if(passwd.size() != 4)
		return QrfeGlobal::TMI_ERROR;

	QByteArray payload;
    payload.append((char) tagId.size());
    payload.append(tagId);
	payload.append((char) mem_bank);
	payload.append((char) (address >> 8));
	payload.append((char) address);
	payload.append(passwd);
	payload.append((char) blockSize);
    payload.append(data.size());
	payload.append(data);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x03, payload, result );
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}


QrfeGlobal::TMIResult QrfeRfePurReader::getTagHandle ( const QString& tagId, QByteArray &handle )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	QByteArray data;
	data.append(epc.size());
	data.append(epc);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x04, data, result );
	handleProtocolHandlerError(res);

	QrfeGlobal::TMIResult tmi_res = translateGen2ProtocolHandlerResult(res);
	if(tmi_res == QrfeGlobal::TMI_OK )
		handle = result.mid(0, 2);

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::readFromHandle ( const QByteArray& handle, uchar mem_bank, ushort address, const QByteArray& passwd, uchar count, QByteArray &data )
{
	if(handle.size() != 2 || passwd.size() != 4)
		return QrfeGlobal::TMI_ERROR;

	QByteArray payload;
	payload.append(handle);
	payload.append((char) mem_bank);
	payload.append((char) (address >> 8));
	payload.append((char) address);
	payload.append(passwd);

	payload.append(count);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x05, payload, result );
	handleProtocolHandlerError(res);

	QrfeGlobal::TMIResult tmi_res = translateGen2ProtocolHandlerResult(res);
	if(tmi_res == QrfeGlobal::TMI_OK )
		data = result.mid(1, (uchar)result.at(0));

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::writeToHandle ( const QByteArray& handle, uchar mem_bank, ushort address, const QByteArray& passwd, const QByteArray& data )
{
	if(handle.size() != 2 || passwd.size() != 4)
		return QrfeGlobal::TMI_ERROR;

	QByteArray payload;
	payload.append(handle);
	payload.append((char) mem_bank);
	payload.append((char) (address >> 8));
	payload.append((char) address);
	payload.append(passwd);
    payload.append(data.size());
	payload.append(data);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x06, payload, result );
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::blockWriteToHandle ( const QByteArray& handle, uchar mem_bank, ushort address, const QByteArray& passwd, uchar blockSize, const QByteArray& data )
{
	if(handle.size() != 2 || passwd.size() != 4)
		return QrfeGlobal::TMI_ERROR;

	QByteArray payload;
	payload.append(handle);
	payload.append((char) mem_bank);
	payload.append((char) (address >> 8));
	payload.append((char) address);
	payload.append(passwd);
	payload.append((char) blockSize);
    payload.append(data.size());
	payload.append(data);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x08, payload, result );
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::customGen2Command ( const QByteArray& handle, const QByteArray& command, const QByteArray& passwd, uchar txBitCount, const QByteArray& txBits, bool& headerBit, QByteArray& rxBytes )
{
	if (command.size() != 2 || passwd.size() != 4)
	{
		trc(3, "Custom Tag Command Handle - NOK - Data");
		return QrfeGlobal::TMI_ERROR;
	}

	uchar txByteCount = (uchar)(txBitCount >> 3);
    if ((txBitCount % 8) > 0)
        txByteCount++;

    if (txByteCount != txBits.size())
    {
        trc(3, "Custom Tag Command Handle - NOK - Data");
        return QrfeGlobal::TMI_ERROR;
    }

	QByteArray payload;
	payload.append(handle);
	payload.append(command);
	payload.append(passwd);
	payload.append(txBitCount);
	payload.append(txBits);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x07, payload, result );
	handleProtocolHandlerError(res);

	QrfeGlobal::TMIResult tmi_res = translateGen2ProtocolHandlerResult(res);
	if(tmi_res == QrfeGlobal::TMI_OK )
	{
		headerBit = (result.at(0) == 1);
		rxBytes = result.mid(2, (uchar)result.at(1));
	}

	return tmi_res;
}


QrfeGlobal::TMIResult QrfeRfePurReader::nxp_setReadProtect ( const QString& tagId, const QByteArray& passwd )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	QByteArray data;
	data.append(epc.size());
	data.append(epc);
	data.append(passwd);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x01, data, result );
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::nxp_resetReadProtect ( const QString& tagId, const QByteArray& passwd )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	QByteArray data;
	data.append(epc.size());
	data.append(epc);
	data.append(passwd);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x02, data, result );
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}

QrfeGlobal::TMIResult QrfeRfePurReader::nxp_sendCmdToCtrler ( const QString& tagId, uchar cmd, const QByteArray& cmdData )
{
	bool ok = false;
	QByteArray epc = QrfeGlobal::stringToBytes(tagId, &ok);
	if(!ok)return QrfeGlobal::TMI_ERROR;

	QByteArray data;
	data.append(epc.size());
	data.append(epc);
	data.append((char) EPCGlobal::MEM_USER);
	data.append((char) 0x00);
	data.append((char) 0x7F);
	data.append(1 + cmdData.size());
	data.append(cmd);
	data.append(cmdData);
	if(cmdData.size()%2 == 0)
		data.append((char)0);

	QByteArray result;

	QrfeGlobal::Result res = m_ph->customTagCommand( 0x03, data, result );
	handleProtocolHandlerError(res);

	return translateGen2ProtocolHandlerResult(res);
}


QrfeTagManipulatorInterface* QrfeRfePurReader::getTagManipulator()
{
	return (QrfeTagManipulatorInterface*) this;
}

QrfeReaderInterface* QrfeRfePurReader::getReaderInterface()
{
    return (QrfeReaderInterface*) this;
}

bool QrfeRfePurReader::tagManipulatorCanReadToEnd()
{
    return true;
}

uint QrfeRfePurReader::tagManipulatorMaxBufferSize()
{
    return 200;
}

void QrfeRfePurReader::getRSSIChannels(QList<QString>& names, QList<long>& minimum, QList<long>& maximum, QList<QString>& units)
{
	foreach(QString name, m_rssiChildNames)
	{
		names << name;
		minimum << m_rssiChildMin;
		maximum << m_rssiChildMax;
		units << "dB";
	}

//	if(m_retreivedValidG)
//	{
//		names << "Power*10";
//		minimum << -100;
//		maximum << 100;
//		units << "dBm";
//	}
}


QString QrfeRfePurReader::getNotificationName(uchar id)
{
	switch (id)
	{
	case NOTIFICATION_ID_ANTENNA_POWER_CHANGED:
		return ("Antenna Power");
	case NOTIFICATION_ID_FREQUENCY_CHANGED:
		return ("Frequency");
	case NOTIFICATION_ID_INVENTORY_ROUND_ENDED:
		return ("Inventory End");
    case NOTIFICATION_ID_LBT_RSSI_VALUE_MEASURED:
        return ("LBT RSSI Value");
    case NOTIFICATION_ID_ANTENNA_CHANNEL_CHANGED:
        return ("Antenna channel changed");
    case NOTIFICATION_ID_ANTENNA_RFP_VALUE_MEASURED:
        return ("RFP Value measured");
    case NOTIFICATION_ID_ANTENNA_RFP_ERROR:
        return ("RFP Error");
    case NOTIFICATION_ID_DEBUG_CODES:
        return ("Debug Code");
    }
	return "";
}

void QrfeRfePurReader::getApplicationInformation( const QByteArray& data, QMap<uint,QPair<QString,QString> >& genericInfo)
{
	QPair<QString,QString> pair;
	pair.first = "APPL-Data";
	pair.second = data.toHex();

	genericInfo.insert(1, pair);

	return;
}


void QrfeRfePurReader::notificationSlot ( const uchar id, const QByteArray& data )
{
    QrfeRfeReader::notificationSlot(id, data);

    switch(id)
    {
    case NOTIFICATION_ID_ANTENNA_POWER_CHANGED:
        if(data.size() != 1)
            return;
        trc(0, "Antenna power changed to " + QString::number(data.at(0)));
        emit notification(NOTIFICATION_ID_ANTENNA_POWER_CHANGED, (bool)data.at(0));
        break;
    case NOTIFICATION_ID_FREQUENCY_CHANGED:
    {
        if(data.size() != 3)
            return;
        uint freq = 0;
        freq |= (((ulong) (uchar) data.at(0)) << 16);
        freq |= (((ulong) (uchar) data.at(1)) << 8);
        freq |=   (ulong) (uchar) data.at(2);
        trc(0, "Frequency changed to " + QString::number(freq));
        emit notification(NOTIFICATION_ID_FREQUENCY_CHANGED, (uint)freq);
        break;
    }
    case NOTIFICATION_ID_INVENTORY_ROUND_ENDED:
        trc(0, "Inventory round ended");
        emit notification(NOTIFICATION_ID_INVENTORY_ROUND_ENDED, true);
        break;
    case NOTIFICATION_ID_LBT_RSSI_VALUE_MEASURED:
    {
        if(data.size() != 2)
            return;
        short rssi = 0;
        rssi |= (((short) (uchar) data.at(0)) << 8);
        rssi |= (((short) (uchar) data.at(1)));
        emit notification(NOTIFICATION_ID_LBT_RSSI_VALUE_MEASURED, rssi);
        break;
    }
    case NOTIFICATION_ID_ANTENNA_CHANNEL_CHANGED:
    {
        if(data.size() != 1)
            return;
        uchar id = 0;
        id = (uchar) data.at(0);

        emit notification(NOTIFICATION_ID_ANTENNA_CHANNEL_CHANGED, (int)id);
        break;
    }
    case NOTIFICATION_ID_ANTENNA_RFP_VALUE_MEASURED:
    {
        if(data.size() != 5)
            return;
        uchar id = 0;
        id = (uchar) data.at(0);
        int rfp = 0;
        rfp |= (((int) (uchar) data.at(1)) << 24);
        rfp |= (((int) (uchar) data.at(2)) << 16);
        rfp |= (((int) (uchar) data.at(3)) << 8);
        rfp |= (((int) (uchar) data.at(4)));

        QByteArray ba;
        QDataStream stream(&ba, QIODevice::WriteOnly);
        stream << id;
        stream << rfp;

        emit notification(NOTIFICATION_ID_ANTENNA_RFP_VALUE_MEASURED, ba);
        break;
    }
    case NOTIFICATION_ID_ANTENNA_RFP_ERROR:
    {
        if(data.size() != 6)
            return;
        uchar id = 0;
        id = (uchar) data.at(0);
        bool error = 0;
        error = (data.at(1) != 0);
        int rfp = 0;
        rfp |= (((int) (uchar) data.at(2)) << 24);
        rfp |= (((int) (uchar) data.at(3)) << 16);
        rfp |= (((int) (uchar) data.at(4)) << 8);
        rfp |= (((int) (uchar) data.at(5)));

        QByteArray ba;
        QDataStream stream(&ba, QIODevice::WriteOnly);
        stream << id;
        stream << error;
        stream << rfp;

        emit notification(NOTIFICATION_ID_ANTENNA_RFP_ERROR, ba);
        break;
    }
    case NOTIFICATION_ID_DEBUG_CODES:
    {
        if(data.size() < 3)
            return;

        uchar id = 0;
        id = (uchar) data.at(0);
        ushort code = 0;
        code |= (((ushort) (uchar) data.at(1)) << 8);
        code |= (((ushort) (uchar) data.at(2)));

        QByteArray info = data.mid(3);

        QByteArray ba;
        QDataStream stream(&ba, QIODevice::WriteOnly);
        stream << id;
        stream << code;
        stream << info;

        trc(0, "DEBUG_CODE: " + QString::number(id) + "-" + QString::number(code) + "-" + info.toHex());

        emit notification(NOTIFICATION_ID_DEBUG_CODES, ba);
        break;
    }
    }
}


void QrfeRfePurReader::acknowledgeAS399xStatusBit()
{
	m_ph->acknowledgeStatusBit(STATUSBIT_AS399X_STATUS_BIT);
}

bool QrfeRfePurReader::filter_statusRegisterChanged(const qulonglong /*statusRegister*/)
{
	return true;
}


QStringList QrfeRfePurReader::statusStrings() const
{
	return QStringList() << QString::number(m_as399xResetCount) + " resets of AS3993x";
}


QByteArray QrfeRfePurReader::interpretRSSItoPercent(const QByteArray& /*tagId*/, QByteArray rssi)
{
	if(rssi.size() != 2)
		return QByteArray();

	uchar q = rssi.at(0);
	uchar i = rssi.at(1);
	uchar rssiVal = qMax( q, i );

	QByteArray res;
	res.append((uchar) ((rssiVal / 32.0) * 100.0));
	res.append(rssi);

//	if(m_retreivedValidG)
//	{
//		double highRSSI = qMax( q, i );
//		double deltaRSSI = q-i;		if(deltaRSSI < 0)deltaRSSI*=-1;
//		double power = (2 * highRSSI) + (10*log10( 1 + pow(10, ( (deltaRSSI * (-1)) / 10)) )) - m_as399xGconstant - 3;
//
//		res.append(power*10);
//	}

	return res;
}

QString QrfeRfePurReader::interpretTriggerId(const QByteArray& /*tagId*/, uchar /*trigger*/)
{
	return "Unknown";
}

QString	QrfeRfePurReader::interpretStateName(const QByteArray& /*tagId*/, ushort /*state*/)
{
	return "Unknown";
}


QrfeGlobal::TMIResult QrfeRfePurReader::translateGen2ProtocolHandlerResult(QrfeGlobal::Result res)
{
	switch(res)
	{
	case QrfeGlobal::RES_OK:
		return QrfeGlobal::TMI_OK;

	case QrfeGlobal::RES_ERROR:
	{
		switch(m_ph->lastReturnCode())
		{
		case RFE_RET_TMI_INSUFFICIENT_POWER:
			return QrfeGlobal::TMI_INSUFFICIENT_POWER;
		case RFE_RET_TMI_MEM_LOCKED:
			return QrfeGlobal::TMI_MEM_LOCKED;
		case RFE_RET_TMI_MEM_OVERRUN:
			return QrfeGlobal::TMI_MEM_OVERRUN;
		case RFE_RET_TMI_TAG_UNREACHABLE:
			return QrfeGlobal::TMI_TAG_UNREACHABLE;
		case RFE_RET_TMI_WRONG_PASSWORD:
			return QrfeGlobal::TMI_WRONG_PASSW;
		default:
			return QrfeGlobal::TMI_ERROR;
		}
		return QrfeGlobal::TMI_ERROR;
	}

	case QrfeGlobal::RES_COMMERROR:
		return QrfeGlobal::TMI_COMMERROR;

	case QrfeGlobal::RES_NORESPONSE:
		return QrfeGlobal::TMI_NORESPONSE;

	default:
		return QrfeGlobal::TMI_ERROR;
	}
}
